#include "CRfePurHiddenSettings.h"

#include <reader/QrfeRfePurReader>

CRfePurHiddenSettings::CRfePurHiddenSettings(QTextEdit* infoEdit, QWidget* parent)
	: ReaderTabWidget(tr("Hidden Settings"), parent)
    , m_reader(0)
{
	ui.setupUi(this);
	setInfoEdit(infoEdit);

	connect(ui.minAttButton,				SIGNAL(clicked()),
			this, 							  SLOT(setMinAtt()));
	connect(ui.freqOffsetButton,			SIGNAL(clicked()),
			this, 							  SLOT(setFrequOffset()));
	connect(ui.defaultFrequencyListButton,	SIGNAL(clicked()),
			this, 							  SLOT(setDefaultFrequList()));
    connect(ui.defaultModSetButton,			SIGNAL(clicked()),
            this, 							  SLOT(setDefaultModSet()));
    connect(ui.antennaCountButton,			SIGNAL(clicked()),
            this, 							  SLOT(setAntennaCount()));
    connect(ui.powerSafeButton,             SIGNAL(clicked()),
            this, 							  SLOT(setPowerSafeSettings()));
    connect(ui.referenceFrequencyButton,    SIGNAL(clicked()),
            this, 							  SLOT(setReferenceFrequency()));


	connect(ui.minAttSlider,	 			SIGNAL(valueChanged(int)),
			this, 							  SLOT(setMinAttLabel(int)));
}

CRfePurHiddenSettings::~CRfePurHiddenSettings()
{

}

bool CRfePurHiddenSettings::init(QrfeReaderInterface* reader)
{
	m_reader = qobject_cast<QrfeRfePurReader*>(reader);
	if( m_reader == 0 )
		return false;

	ulong sw_rev = m_reader->softwareRevision();
	sw_rev &= 0x0000FFFF;
	if(sw_rev < 0x0027)
		return false;

	clearInfo();

	return true;
}

void CRfePurHiddenSettings::deinit()
{
	m_reader = 0;
}

void CRfePurHiddenSettings::activated()
{
    if(m_reader == 0)
        return;

	clearInfo();
	reloadSettings();
}

void CRfePurHiddenSettings::reloadSettings()
{
	uchar minAtt = 0;
	if(m_reader->minimumAttenuation(minAtt) != QrfeGlobal::RES_OK)
	{
		ui.minAttSlider->setEnabled(false);
		ui.minAttButton->setEnabled(false);
	}
	else
	{
		ui.minAttSlider->setMaximum(19);

		ui.minAttSlider->setEnabled(true);
		ui.minAttButton->setEnabled(true);
		ui.minAttSlider->setValue(minAtt);
	}

	long freqOffset = 0;
	if(m_reader->frequencyOffset(freqOffset) != QrfeGlobal::RES_OK)
	{
		ui.freqOffsetBox->setEnabled(false);
		ui.freqOffsetButton->setEnabled(false);
	}
	else
	{
		ui.freqOffsetBox->setEnabled(true);
		ui.freqOffsetButton->setEnabled(true);
		ui.freqOffsetBox->setValue(freqOffset);
	}

	uchar defaultFreqListIndex = 0;
	if(m_reader->defaultFrequencyListIndex(defaultFreqListIndex) != QrfeGlobal::RES_OK)
	{
		ui.defaultFrequencyListBox->setEnabled(false);
		ui.defaultFrequencyListButton->setEnabled(false);
	}
	else
	{
		ui.defaultFrequencyListBox->setEnabled(true);
		ui.defaultFrequencyListButton->setEnabled(true);
		ui.defaultFrequencyListBox->setCurrentIndex(defaultFreqListIndex);
	}

	ulong defaultModSet = 0;
	if(m_reader->defaultModulatorSettings(defaultModSet) != QrfeGlobal::RES_OK)
	{
		ui.defaultModSetButton->setEnabled(false);
		ui.defaultModSetPrAskCheckBox->setEnabled(false);
		ui.defaultModSetLowPassCheckBox->setEnabled(false);
	}
	else
	{
		ui.defaultModSetButton->setEnabled(true);
		ui.defaultModSetPrAskCheckBox->setEnabled(true);
		ui.defaultModSetLowPassCheckBox->setEnabled(true);

		ui.defaultModSetPrAskCheckBox->setChecked( (defaultModSet & 0x00000001) != 0 );
		ui.defaultModSetLowPassCheckBox->setChecked( (defaultModSet & 0x00000002) != 0 );
	}


    uchar maxAntennaCount = 0;
    if(m_reader->maxAntennaCount(maxAntennaCount) != QrfeGlobal::RES_OK)
    {
        ui.antennaCountBox->setEnabled(false);
        ui.antennaCountButton->setEnabled(false);
    }
    else
    {
        m_reader->readSettings();
        ui.antennaCountBox->setEnabled(true);
        ui.antennaCountButton->setEnabled(true);
		ui.antennaCountBox->setMaximum(maxAntennaCount);
		ui.antennaCountBox->setValue(m_reader->antennaCount());
    }


    bool required = false;
    ushort minTime = 0;
    ushort maxCount = 0;
    if(m_reader->powerSafeRequiredSettings(required, minTime, maxCount) != QrfeGlobal::RES_OK)
    {
        ui.powerSafe_minSetOn_checkBox->setEnabled(false);
        ui.powerSafe_minTime_spinBox->setEnabled(false);
        ui.powerSafe_maxCount_spinBox->setEnabled(false);
        ui.powerSafeButton->setEnabled(false);
    }
    else
    {
        ui.powerSafe_minSetOn_checkBox->setEnabled(true);
        ui.powerSafe_minTime_spinBox->setEnabled(true);
        ui.powerSafe_maxCount_spinBox->setEnabled(true);
        ui.powerSafeButton->setEnabled(true);
        ui.powerSafe_minSetOn_checkBox->setChecked(required);
        ui.powerSafe_minTime_spinBox->setValue(minTime);
        ui.powerSafe_maxCount_spinBox->setValue(maxCount);
    }

    uchar referenceFrequency = 0;
    if(m_reader->referenceFrequency(referenceFrequency) != QrfeGlobal::RES_OK)
    {
        ui.referenceFrequencyButton->setEnabled(false);
        ui.referenceFrequencyListBox->setEnabled(false);
    }
    else
    {
        ui.referenceFrequencyButton->setEnabled(true);
        ui.referenceFrequencyListBox->setEnabled(true);

        ui.antennaCountBox->setValue(refFrequencyToIndex(referenceFrequency));
    }


    infoAppend("-- Reload settings --");
	showOK();
}

void CRfePurHiddenSettings::setMinAtt()
{
	if(m_reader == 0)
		return;

	uchar minAtt = ui.minAttSlider->value();
	QrfeGlobal::Result res = m_reader->setMinimumAttenuation(minAtt);
	if(res == QrfeGlobal::RES_OK)	{
		infoSetOK("-- Set minimum attenuation to " + QString::number(minAtt) + " --");
	}
	else {
		infoSetError("-- Could not set minimum attenuation --");
	}

	reloadSettings();
}

void CRfePurHiddenSettings::setFrequOffset()
{
	if(m_reader == 0)
		return;

	long freqOffset = ui.freqOffsetBox->value();
	QrfeGlobal::Result res = m_reader->setFrequencyOffset(freqOffset);
	if(res == QrfeGlobal::RES_OK)	{
		infoSetOK("-- Set frequency offset to " + QString::number(freqOffset) + " --");
	}
	else {
		infoSetError("-- Could not set frequency offset --");
	}

	reloadSettings();
}

void CRfePurHiddenSettings::setDefaultFrequList()
{
	if(m_reader == 0)
		return;

	uchar freqListIndex = ui.defaultFrequencyListBox->currentIndex();
	QrfeGlobal::Result res = m_reader->setDefaultFrequencyListIndex(freqListIndex);
	if(res == QrfeGlobal::RES_OK)	{
		infoSetOK("-- Set default frequency list to " + ui.defaultFrequencyListBox->currentText() + " --");
	}
	else {
		infoSetError("-- Could not set default frequency list index --");
	}

	reloadSettings();
}

void CRfePurHiddenSettings::setDefaultModSet()
{
	if(m_reader == 0)
		return;

	ulong modSet = 0;

	modSet |= (ui.defaultModSetPrAskCheckBox->isChecked()) ? 0x00000001 : 0;
	modSet |= (ui.defaultModSetLowPassCheckBox->isChecked()) ? 0x00000002 : 0;

	QrfeGlobal::Result res = m_reader->setDefaultModulatorSettings(modSet);
	if(res == QrfeGlobal::RES_OK)	{
		infoSetOK("-- Set default modulator settings to " + QString("%1").arg(modSet, 8, 16, QChar('0')) + " --");
	}
	else {
		infoSetError("-- Could not set default modulator settings --");
	}

    reloadSettings();
}

void CRfePurHiddenSettings::setAntennaCount()
{
    if(m_reader == 0)
        return;

    uchar antennaCount = ui.antennaCountBox->value();
    QrfeGlobal::Result res = m_reader->setAntennaCount(antennaCount);
    if(res == QrfeGlobal::RES_OK)	{
        infoSetOK("-- Set antenna count to " + QString::number(antennaCount) + " --");
    }
    else {
        infoSetError("-- Could not set antenna count --");
    }

    reloadSettings();
}

void CRfePurHiddenSettings::setPowerSafeSettings()
{
    if(m_reader == 0)
        return;

    bool required = ui.powerSafe_minSetOn_checkBox->isChecked();
    ushort minTime = ui.powerSafe_minTime_spinBox->value();
    ushort maxCount = ui.powerSafe_maxCount_spinBox->value();
    QrfeGlobal::Result res = m_reader->setPowerSafeRequiredSettings(required, minTime, maxCount);
    if(res == QrfeGlobal::RES_OK)	{
        infoSetOK("-- Set power safe minimum settings to " + QString::number(required) + " - " + QString::number(minTime) + " - " + QString::number(maxCount) + " --");
    }
    else {
        infoSetError("-- Could not set power safe minimum settings --");
    }

    reloadSettings();
}

void CRfePurHiddenSettings::setReferenceFrequency()
{
    if(m_reader == 0)
        return;

    uchar refFreq = indexToRefefreqeuncy(ui.referenceFrequencyListBox->currentIndex());
    QrfeGlobal::Result res = m_reader->setReferenceFrequency(refFreq);
    if(res == QrfeGlobal::RES_OK)	{
        infoSetOK("-- Set Reference Freqeuncy count to " + QString::number(refFreq) + " --");
    }
    else {
        infoSetError("-- Could not set Reference Freqeuncy --");
    }

    reloadSettings();
}



void CRfePurHiddenSettings::setMinAttLabel(int val)
{
    ui.minAttLabel->setText(QString::number(val) + " steps");
}

int CRfePurHiddenSettings::refFrequencyToIndex(uchar refFreq)
{
   switch(refFreq)
   {
   case 25:
       return 0;
   case 50:
       return 1;
   case 100:
       return 2;
   case 200:
       return 3;
   case 250:
       return 4;
   default:
       return 5;
   }
}

uchar CRfePurHiddenSettings::indexToRefefreqeuncy(int index)
{
    switch(index)
    {
    case 0:
        return 25;
    case 1:
        return 50;
    case 2:
        return 100;
    case 3:
        return 200;
    case 4:
        return 250;
    default:
        return 25;
    }
}


